<?php
/**
 * Plugin Name: SmartBanner - Location-Based Pricing
 * Plugin URI: https://smartbanner.pro
 * Description: Display location-based pricing banners with automatic regional discounts. Increase conversions with PPP (Purchasing Power Parity) pricing.
 * Version: 1.0.0
 * Author: SmartBanner
 * Author URI: https://smartbanner.pro
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: smartbanner
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('SMARTBANNER_VERSION', '1.0.0');
define('SMARTBANNER_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SMARTBANNER_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Main SmartBanner Plugin Class
 */
class SmartBanner_Plugin {

    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_head', array($this, 'inject_banner_script'));
        add_action('admin_enqueue_scripts', array($this, 'admin_styles'));
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_settings_link'));
    }

    /**
     * Add settings link to plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="options-general.php?page=smartbanner">' . __('Settings', 'smartbanner') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            __('SmartBanner Settings', 'smartbanner'),
            __('SmartBanner', 'smartbanner'),
            'manage_options',
            'smartbanner',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        // Register settings
        register_setting('smartbanner_settings', 'smartbanner_api_key', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ));

        register_setting('smartbanner_settings', 'smartbanner_enabled', array(
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default' => true
        ));

        register_setting('smartbanner_settings', 'smartbanner_position', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_position'),
            'default' => 'top'
        ));

        register_setting('smartbanner_settings', 'smartbanner_theme', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_theme'),
            'default' => 'auto'
        ));

        register_setting('smartbanner_settings', 'smartbanner_delay', array(
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'default' => 0
        ));

        register_setting('smartbanner_settings', 'smartbanner_close_button', array(
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default' => true
        ));

        register_setting('smartbanner_settings', 'smartbanner_bg_color', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_hex_color',
            'default' => ''
        ));

        register_setting('smartbanner_settings', 'smartbanner_text_color', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_hex_color',
            'default' => ''
        ));

        register_setting('smartbanner_settings', 'smartbanner_accent_color', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_hex_color',
            'default' => ''
        ));

        register_setting('smartbanner_settings', 'smartbanner_exclude_pages', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_textarea_field',
            'default' => ''
        ));

        // Add settings sections
        add_settings_section(
            'smartbanner_main_section',
            __('Main Settings', 'smartbanner'),
            array($this, 'main_section_callback'),
            'smartbanner'
        );

        add_settings_section(
            'smartbanner_appearance_section',
            __('Appearance', 'smartbanner'),
            array($this, 'appearance_section_callback'),
            'smartbanner'
        );

        add_settings_section(
            'smartbanner_advanced_section',
            __('Advanced', 'smartbanner'),
            array($this, 'advanced_section_callback'),
            'smartbanner'
        );

        // Add settings fields - Main
        add_settings_field(
            'smartbanner_api_key',
            __('API Key', 'smartbanner'),
            array($this, 'api_key_field_callback'),
            'smartbanner',
            'smartbanner_main_section'
        );

        add_settings_field(
            'smartbanner_enabled',
            __('Enable Banner', 'smartbanner'),
            array($this, 'enabled_field_callback'),
            'smartbanner',
            'smartbanner_main_section'
        );

        // Add settings fields - Appearance
        add_settings_field(
            'smartbanner_position',
            __('Banner Position', 'smartbanner'),
            array($this, 'position_field_callback'),
            'smartbanner',
            'smartbanner_appearance_section'
        );

        add_settings_field(
            'smartbanner_theme',
            __('Theme', 'smartbanner'),
            array($this, 'theme_field_callback'),
            'smartbanner',
            'smartbanner_appearance_section'
        );

        add_settings_field(
            'smartbanner_colors',
            __('Custom Colors', 'smartbanner'),
            array($this, 'colors_field_callback'),
            'smartbanner',
            'smartbanner_appearance_section'
        );

        add_settings_field(
            'smartbanner_close_button',
            __('Show Close Button', 'smartbanner'),
            array($this, 'close_button_field_callback'),
            'smartbanner',
            'smartbanner_appearance_section'
        );

        // Add settings fields - Advanced
        add_settings_field(
            'smartbanner_delay',
            __('Display Delay', 'smartbanner'),
            array($this, 'delay_field_callback'),
            'smartbanner',
            'smartbanner_advanced_section'
        );

        add_settings_field(
            'smartbanner_exclude_pages',
            __('Exclude Pages', 'smartbanner'),
            array($this, 'exclude_pages_field_callback'),
            'smartbanner',
            'smartbanner_advanced_section'
        );
    }

    /**
     * Sanitize position value
     */
    public function sanitize_position($value) {
        $allowed = array('top', 'bottom', 'modal');
        return in_array($value, $allowed) ? $value : 'top';
    }

    /**
     * Sanitize theme value
     */
    public function sanitize_theme($value) {
        $allowed = array('auto', 'light', 'dark');
        return in_array($value, $allowed) ? $value : 'auto';
    }

    /**
     * Section callbacks
     */
    public function main_section_callback() {
        echo '<p>' . esc_html__('Configure your SmartBanner integration. Get your API key from your SmartBanner dashboard.', 'smartbanner') . '</p>';
    }

    public function appearance_section_callback() {
        echo '<p>' . esc_html__('Customize how the banner looks on your site.', 'smartbanner') . '</p>';
    }

    public function advanced_section_callback() {
        echo '<p>' . esc_html__('Advanced configuration options.', 'smartbanner') . '</p>';
    }

    /**
     * Field callbacks
     */
    public function api_key_field_callback() {
        $value = get_option('smartbanner_api_key', '');
        ?>
        <input type="text"
               name="smartbanner_api_key"
               value="<?php echo esc_attr($value); ?>"
               class="regular-text"
               placeholder="sb_live_xxxxxxxxxxxxx"
        />
        <p class="description">
            <?php echo wp_kses(
                __('Enter your SmartBanner API key. Find it in your <a href="https://smartbanner.pro/dashboard/settings" target="_blank">SmartBanner Dashboard</a>.', 'smartbanner'),
                array('a' => array('href' => array(), 'target' => array()))
            ); ?>
        </p>
        <?php
    }

    public function enabled_field_callback() {
        $value = get_option('smartbanner_enabled', true);
        ?>
        <label>
            <input type="checkbox"
                   name="smartbanner_enabled"
                   value="1"
                   <?php checked($value, true); ?>
            />
            <?php esc_html_e('Display the SmartBanner on your site', 'smartbanner'); ?>
        </label>
        <?php
    }

    public function position_field_callback() {
        $value = get_option('smartbanner_position', 'top');
        ?>
        <select name="smartbanner_position">
            <option value="top" <?php selected($value, 'top'); ?>><?php esc_html_e('Top Bar', 'smartbanner'); ?></option>
            <option value="bottom" <?php selected($value, 'bottom'); ?>><?php esc_html_e('Bottom Bar', 'smartbanner'); ?></option>
            <option value="modal" <?php selected($value, 'modal'); ?>><?php esc_html_e('Modal Popup', 'smartbanner'); ?></option>
        </select>
        <p class="description"><?php esc_html_e('Choose where the banner appears on your site.', 'smartbanner'); ?></p>
        <?php
    }

    public function theme_field_callback() {
        $value = get_option('smartbanner_theme', 'auto');
        ?>
        <select name="smartbanner_theme">
            <option value="auto" <?php selected($value, 'auto'); ?>><?php esc_html_e('Auto (matches system)', 'smartbanner'); ?></option>
            <option value="light" <?php selected($value, 'light'); ?>><?php esc_html_e('Light', 'smartbanner'); ?></option>
            <option value="dark" <?php selected($value, 'dark'); ?>><?php esc_html_e('Dark', 'smartbanner'); ?></option>
        </select>
        <p class="description"><?php esc_html_e('Choose the color theme for the banner.', 'smartbanner'); ?></p>
        <?php
    }

    public function colors_field_callback() {
        $bg_color = get_option('smartbanner_bg_color', '');
        $text_color = get_option('smartbanner_text_color', '');
        $accent_color = get_option('smartbanner_accent_color', '');
        ?>
        <div class="smartbanner-color-fields">
            <div class="smartbanner-color-field">
                <label><?php esc_html_e('Background Color', 'smartbanner'); ?></label>
                <input type="text"
                       name="smartbanner_bg_color"
                       value="<?php echo esc_attr($bg_color); ?>"
                       class="smartbanner-color-picker"
                       placeholder="#6366f1"
                />
            </div>
            <div class="smartbanner-color-field">
                <label><?php esc_html_e('Text Color', 'smartbanner'); ?></label>
                <input type="text"
                       name="smartbanner_text_color"
                       value="<?php echo esc_attr($text_color); ?>"
                       class="smartbanner-color-picker"
                       placeholder="#ffffff"
                />
            </div>
            <div class="smartbanner-color-field">
                <label><?php esc_html_e('Accent Color', 'smartbanner'); ?></label>
                <input type="text"
                       name="smartbanner_accent_color"
                       value="<?php echo esc_attr($accent_color); ?>"
                       class="smartbanner-color-picker"
                       placeholder="#fbbf24"
                />
            </div>
        </div>
        <p class="description"><?php esc_html_e('Leave empty to use the default theme colors. Enter hex colors (e.g., #6366f1).', 'smartbanner'); ?></p>
        <?php
    }

    public function close_button_field_callback() {
        $value = get_option('smartbanner_close_button', true);
        ?>
        <label>
            <input type="checkbox"
                   name="smartbanner_close_button"
                   value="1"
                   <?php checked($value, true); ?>
            />
            <?php esc_html_e('Allow visitors to dismiss the banner', 'smartbanner'); ?>
        </label>
        <?php
    }

    public function delay_field_callback() {
        $value = get_option('smartbanner_delay', 0);
        ?>
        <input type="number"
               name="smartbanner_delay"
               value="<?php echo esc_attr($value); ?>"
               min="0"
               max="30000"
               step="100"
               class="small-text"
        /> <?php esc_html_e('milliseconds', 'smartbanner'); ?>
        <p class="description"><?php esc_html_e('Delay before showing the banner. 1000ms = 1 second.', 'smartbanner'); ?></p>
        <?php
    }

    public function exclude_pages_field_callback() {
        $value = get_option('smartbanner_exclude_pages', '');
        ?>
        <textarea name="smartbanner_exclude_pages"
                  rows="4"
                  cols="50"
                  class="large-text"
                  placeholder="/checkout&#10;/cart&#10;/thank-you"
        ><?php echo esc_textarea($value); ?></textarea>
        <p class="description"><?php esc_html_e('Enter URL paths to exclude (one per line). The banner will not appear on these pages.', 'smartbanner'); ?></p>
        <?php
    }

    /**
     * Admin styles
     */
    public function admin_styles($hook) {
        if ('settings_page_smartbanner' !== $hook) {
            return;
        }

        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');

        wp_add_inline_style('wp-color-picker', '
            .smartbanner-color-fields {
                display: flex;
                gap: 20px;
                flex-wrap: wrap;
            }
            .smartbanner-color-field {
                display: flex;
                flex-direction: column;
                gap: 5px;
            }
            .smartbanner-color-field label {
                font-weight: 500;
                font-size: 13px;
            }
            .smartbanner-header {
                background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
                color: white;
                padding: 30px;
                margin: -10px -20px 20px -20px;
                border-radius: 0;
            }
            .smartbanner-header h1 {
                color: white;
                margin: 0 0 10px 0;
            }
            .smartbanner-header p {
                opacity: 0.9;
                margin: 0;
            }
            .smartbanner-status {
                display: inline-flex;
                align-items: center;
                gap: 8px;
                padding: 8px 16px;
                border-radius: 20px;
                font-size: 13px;
                font-weight: 500;
                margin-top: 15px;
            }
            .smartbanner-status.active {
                background: rgba(34, 197, 94, 0.2);
                color: #22c55e;
            }
            .smartbanner-status.inactive {
                background: rgba(239, 68, 68, 0.2);
                color: #ef4444;
            }
            .smartbanner-card {
                background: white;
                border: 1px solid #e5e7eb;
                border-radius: 8px;
                padding: 20px;
                margin-bottom: 20px;
            }
            .smartbanner-card h2 {
                margin-top: 0;
                padding-bottom: 15px;
                border-bottom: 1px solid #e5e7eb;
            }
        ');

        wp_add_inline_script('wp-color-picker', '
            jQuery(document).ready(function($) {
                $(".smartbanner-color-picker").wpColorPicker();
            });
        ');
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $api_key = get_option('smartbanner_api_key', '');
        $enabled = get_option('smartbanner_enabled', true);
        $is_configured = !empty($api_key) && strpos($api_key, 'sb_') === 0;

        ?>
        <div class="wrap">
            <div class="smartbanner-header">
                <h1><?php esc_html_e('SmartBanner', 'smartbanner'); ?></h1>
                <p><?php esc_html_e('Location-based pricing banners to increase your global conversions.', 'smartbanner'); ?></p>
                <?php if ($is_configured && $enabled): ?>
                    <div class="smartbanner-status active">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <?php esc_html_e('Banner is active', 'smartbanner'); ?>
                    </div>
                <?php elseif ($is_configured && !$enabled): ?>
                    <div class="smartbanner-status inactive">
                        <span class="dashicons dashicons-warning"></span>
                        <?php esc_html_e('Banner is disabled', 'smartbanner'); ?>
                    </div>
                <?php else: ?>
                    <div class="smartbanner-status inactive">
                        <span class="dashicons dashicons-warning"></span>
                        <?php esc_html_e('API key required', 'smartbanner'); ?>
                    </div>
                <?php endif; ?>
            </div>

            <form action="options.php" method="post">
                <?php
                settings_fields('smartbanner_settings');
                do_settings_sections('smartbanner');
                submit_button(__('Save Settings', 'smartbanner'));
                ?>
            </form>

            <div class="smartbanner-card">
                <h2><?php esc_html_e('Need Help?', 'smartbanner'); ?></h2>
                <p><?php esc_html_e('Visit our documentation for setup guides, customization options, and troubleshooting.', 'smartbanner'); ?></p>
                <p>
                    <a href="https://smartbanner.pro/docs" target="_blank" class="button">
                        <?php esc_html_e('View Documentation', 'smartbanner'); ?>
                    </a>
                    <a href="https://smartbanner.pro/dashboard" target="_blank" class="button">
                        <?php esc_html_e('Go to Dashboard', 'smartbanner'); ?>
                    </a>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * Inject banner script into frontend
     */
    public function inject_banner_script() {
        // Check if enabled
        $enabled = get_option('smartbanner_enabled', true);
        if (!$enabled) {
            return;
        }

        // Check API key
        $api_key = get_option('smartbanner_api_key', '');
        if (empty($api_key) || strpos($api_key, 'sb_') !== 0) {
            return;
        }

        // Check excluded pages
        $exclude_pages = get_option('smartbanner_exclude_pages', '');
        if (!empty($exclude_pages)) {
            $current_path = wp_parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
            $excluded = array_filter(array_map('trim', explode("\n", $exclude_pages)));
            foreach ($excluded as $path) {
                if (!empty($path) && strpos($current_path, $path) === 0) {
                    return;
                }
            }
        }

        // Don't show in admin
        if (is_admin()) {
            return;
        }

        // Get settings
        $position = get_option('smartbanner_position', 'top');
        $theme = get_option('smartbanner_theme', 'auto');
        $delay = get_option('smartbanner_delay', 0);
        $close_button = get_option('smartbanner_close_button', true);
        $bg_color = get_option('smartbanner_bg_color', '');
        $text_color = get_option('smartbanner_text_color', '');
        $accent_color = get_option('smartbanner_accent_color', '');

        // Build data attributes
        $attrs = array(
            'data-key' => esc_attr($api_key),
            'data-position' => esc_attr($position),
            'data-theme' => esc_attr($theme),
        );

        if ($delay > 0) {
            $attrs['data-delay'] = intval($delay);
        }

        if (!$close_button) {
            $attrs['data-close-button'] = 'false';
        }

        if (!empty($bg_color)) {
            $attrs['data-bg-color'] = esc_attr($bg_color);
        }

        if (!empty($text_color)) {
            $attrs['data-text-color'] = esc_attr($text_color);
        }

        if (!empty($accent_color)) {
            $attrs['data-accent-color'] = esc_attr($accent_color);
        }

        // Build attributes string
        $attr_string = '';
        foreach ($attrs as $key => $value) {
            $attr_string .= ' ' . $key . '="' . $value . '"';
        }

        // Output script
        echo '<script src="https://smartbanner.pro/embed/v1.js"' . $attr_string . ' async></script>' . "\n";
    }
}

// Initialize plugin
SmartBanner_Plugin::get_instance();

/**
 * Activation hook
 */
function smartbanner_activate() {
    // Set default options
    add_option('smartbanner_enabled', true);
    add_option('smartbanner_position', 'top');
    add_option('smartbanner_theme', 'auto');
    add_option('smartbanner_delay', 0);
    add_option('smartbanner_close_button', true);
}
register_activation_hook(__FILE__, 'smartbanner_activate');

/**
 * Deactivation hook
 */
function smartbanner_deactivate() {
    // Optionally clean up options
    // Keeping them for now in case user reactivates
}
register_deactivation_hook(__FILE__, 'smartbanner_deactivate');
